// AreasManager.cpp: implementation of the CAreasManager class.
//
//////////////////////////////////////////////////////////////////////

#include "pch.h"
#include "misc.hpp"
#include "Adrenalin.h"
#include "ConfigWriter.h"
#include "LinksManager.h"
#include "AreasManager.h"

#include "libIncludes\MiniSAX.h"
#include "AreasSAX.h"

using	miniSAX::CDataSource;
using	miniSAX::CMiniSAXParser;
using	miniSAX::CMiniSAXException;


LPCTSTR		AREAS_CFG_NAME = TEXT("areas.xml");


CAreasManager::CAreasManager()
	: areasChanged(false),
	  loaded(false)
{
}


bool isGroupLess(PAreaGroup group1, PAreaGroup group2) {
	return ( lstrcmpi( group1->getName().c_str(), group2->getName().c_str() ) == -1);
}


bool isAreaLess(PArea area1, PArea area2) {
	return ( lstrcmpi( area1->getName().c_str(), area2->getName().c_str() ) == -1);
}


void CAreasManager::load() throw(CBadConfigException)
{
	// we want to load only once
	loaded = true;

    TCHAR		areasCfgFullName[MAX_PATH];

    lstrcpy( areasCfgFullName, CAdrenalin::getStartupPath() );
    lstrcat( areasCfgFullName, AREAS_CFG_NAME );

	try {
		CMiniSAXParser			parser;
		CAreasContentHandler	areasContentHandler( &parser );

		CDataSource		dataSource( areasCfgFullName );

		parser.setContentHandler( &areasContentHandler );

		parser.parse( &dataSource );

	} catch (miniSAX::CIOException& ioEx) {
		throw CBadConfigException( tstring("Failed to read 'areas.xml': ") + ioEx.getMessage() );
	} catch (CMiniSAXException& saxEx) {
		// prepare exception message
		tstring	msg("Failed to read 'areas.xml':");
		if (saxEx.getColumn() >= 0 && saxEx.getRow() >= 0) {
			char	buffer[256];
			sprintf( buffer, " [col:%d,row:%d] ", saxEx.getColumn(), saxEx.getRow() );
			msg += buffer;
		} else {
			msg += " [no position info] ";
		}
		msg += saxEx.getMessage();

		throw CBadConfigException( msg );
	}

	// sort groups
	sort( groups.begin(), groups.end(), isGroupLess );

	// sort areas
	sort( areas.begin(), areas.end(), isAreaLess );
}


CAreasManager::~CAreasManager()
{
}


PAreaGroup CAreasManager::getGroup(int index) {
	return groups[index];
}

/*
bool CAreasManager::parseAreaAccess(LPCTSTR s, PArea area, tstring& group) {
    int  i;

    i = 0;
	int	priority = 0;
    while (_istdigit(*s) && i < 4) {
        priority *= 10;
        priority += (*s - TEXT('0'));
        s++;
        i++;
    }
    if (priority > 1024)
        return (false);
    
    area->setPriority( priority );
    
    if (*s == TEXT('\0')) {
        group = TEXT("");
        return (true);
    }
    
    if (*s != TEXT('/'))
        return (false);
    
    s++;
    
    if (!_istalpha(*s))
        return (false);
    
	group = TEXT("");
	group += (*s);
    
    s++;
    
    return (*s == TEXT('\0'));
}
*/


PAreaGroup CAreasManager::getGroup(const tstring& groupName) {
	for (vector<PAreaGroup>::iterator	it = groups.begin(); it != groups.end(); it++) {
		PAreaGroup	group = (*it);
		if (strcmpi( group->getName().c_str(), groupName.c_str() ) == 0) {
			return group;
		}
	}
	return NULL;
}


PArea CAreasManager::getArea(const tstring& areaName) {
	for (vector<PArea>::iterator	it = areas.begin(); it != areas.end(); it++) {
		PArea	area = (*it);
		if (strcmpi( area->getName().c_str(), areaName.c_str() ) == 0) {
			return area;
		}
	}
	return NULL;
}


void CAreasManager::saveAreas() {
    TCHAR	cfgName[MAX_PATH];

    lstrcpy( cfgName, CAdrenalin::getStartupPath() );
    lstrcat( cfgName, AREAS_CFG_NAME );

	CConfigWriter	out( cfgName );

	// TODO: correctly process '<', '>' and other symbols in data

	// prepare manipulator for entities
	CConfigWriter::CEntityResolver	er;
	CConfigWriter::COemManipulator	oem;

	try {
		out << "<areas>\n";

		for (CGroupIterator	groupIter = groupBegin(); groupIter != groupEnd();
		     ++groupIter)
		{
			PAreaGroup	group = (*groupIter);
			
			// print group name
			out << "    <group name=\"" << er << oem << group->getName() << "\"";
			// print optional group description
			if (group->getDescription().size() > 0) {
				out << "\n           description=\"" << er << oem << group->getDescription() << "\"";
			}
			out << ">\n";

			CAreaIterator	endOfAreas = areaEnd( group );
			for (CAreaIterator areaIter = areaBegin( group ); areaIter != endOfAreas; ++areaIter) {
				PArea	area = (*areaIter);
				// print area name
				out << "        <area name=\"" << er << oem << area->getName() << "\"";
				// print optional group description
				if (area->getDescription().size() > 0) {
					out << "\n              description=\"" << er << oem << area->getDescription() << "\"";
				}
				// print area path
				out << "\n              path=\"" << er << area->getPath() << "\"";
				// print optional send priority
				if (area->getSendPriority() > 0) {
					out << "\n              sendPriority=\"" << area->getSendPriority() << "\"";
				}
				// print optional receive priority
				if (area->getReceivePriority() > 0) {
					out << "\n              receivePriority=\"" << area->getReceivePriority() << "\"";
				}
				out << ">\n";
				// print flags if present
				if (area->hasFlag( CArea::FLAG_PASSTHROUGH )) {
					out << "            <passthrough />\n";
				}
				if (area->hasFlag( CArea::FLAG_VETO_MANUAL_PURGE )) {
					out << "            <vetoManualPurge />\n";
				}
				if (area->hasFlag( CArea::FLAG_OPEN_FOR_RESEND )) {
					out << "            <openForResend />\n";
				}
				// print outbound if defined
				if (area->isOutboundDefined()) {
					out << "            <outbound type=\"";
					switch (area->getOutboundType()) {
						case OT_BINKLEY:
							out << "binkley";
							break;
						case OT_FILE_BOXES:
							out << "fileBoxes";
							break;
						default:
							throw "unknown outbound type";
					}
					out << "\" />\n";
				}
				// print links
				int	tiesCount = area->getTiesCount();
				for (int i = 0; i < tiesCount; i++) {
					TiePtr	tie = area->getTie( i );
					out << "            <linked";
					if (tie->getRouteType() != ROUTE_NONE) {
						out << " routing=\"";
						switch (tie->getRouteType()) {
						case ROUTE_SEND:
							out << "send";
							break;
						case ROUTE_RECEIVE:
							out << "receive";
							break;
						case ROUTE_BIDIRECT:
							out << "bidirectional";
							break;
						}
						out << "\"";
					}
					out << ">" << er << tie->getLink()->getAddress().toShortestString( g_vars.m_addrMain ) << "</linked>\n";
				}
				out << "        </area>\n";
			}

			out << "    </group>\n";
		}		

		out << "</areas>\n";
	} catch (...) {
		out.cancel();
		throw;
	}

	out.close();
}


/*******************************
  CAreasManager::CGroupIterator
 *******************************/


CAreasManager::CGroupIterator::CGroupIterator() {
	manager = NULL;
	index   = -1;
}


CAreasManager::CGroupIterator::CGroupIterator(CAreasManager* aManager, int anIndex) {
	manager = aManager;
	index   = anIndex;
}


CAreasManager::CGroupIterator::CGroupIterator(const CAreasManager::CGroupIterator& instance) {
	manager = instance.manager;
	index   = instance.index;
}


CAreasManager::CGroupIterator& CAreasManager::CGroupIterator::operator =(const CAreasManager::CGroupIterator& instance) {
	manager = instance.manager;
	index   = instance.index;

	return (*this);
}


bool CAreasManager::CGroupIterator::operator ==(const CAreasManager::CGroupIterator& instance) {
	return (manager == instance.manager && index == instance.index);
}


bool CAreasManager::CGroupIterator::operator !=(const CAreasManager::CGroupIterator& instance) {
	return (manager != instance.manager || index != instance.index);
}


CAreasManager::CGroupIterator& CAreasManager::CGroupIterator::operator ++() {
	if (index >= 0) {
		++index;
		if (index >= manager->getGroupsCount()) {
			index = -1;
		}
	}

	return (*this);
}


PAreaGroup CAreasManager::CGroupIterator::operator *() {
	return manager->getGroup( index );
}


CAreasManager::CGroupIterator CAreasManager::groupBegin() {
	if (groups.size() == 0)
		return groupEnd();

	CAreasManager::CGroupIterator	it( this, 0 );

	return it;
}


CAreasManager::CGroupIterator CAreasManager::groupEnd() {
	CAreasManager::CGroupIterator	it( this, -1 );

	return it;
}


/******************************
  CAreasManager::CAreaIterator
 ******************************/


CAreasManager::CAreaIterator::CAreaIterator() {
	manager   = NULL;
	group     = NULL;
	areaIndex = -1;
}


CAreasManager::CAreaIterator::CAreaIterator(CAreasManager* aManager, PAreaGroup aGroup, int anAreaIndex) {
	manager   = aManager;
	group     = aGroup;
	areaIndex = anAreaIndex;
}


CAreasManager::CAreaIterator::CAreaIterator(const CAreasManager::CAreaIterator& instance) {
	manager   = instance.manager;
	group     = instance.group;
	areaIndex = instance.areaIndex;
}


CAreasManager::CAreaIterator& CAreasManager::CAreaIterator::operator =(const CAreasManager::CAreaIterator& instance) {
	manager   = instance.manager;
	group     = instance.group;
	areaIndex = instance.areaIndex;

	return (*this);
}


bool CAreasManager::CAreaIterator::operator ==(const CAreasManager::CAreaIterator& instance) {
	return (manager == instance.manager && group == instance.group && areaIndex == instance.areaIndex);
}


bool CAreasManager::CAreaIterator::operator !=(const CAreasManager::CAreaIterator& instance) {
	return (manager != instance.manager || group != instance.group || areaIndex != instance.areaIndex);
}


CAreasManager::CAreaIterator& CAreasManager::CAreaIterator::operator ++() {
	if (areaIndex >= 0) {
		if (group != NULL) {
			// find next area from the same group
			for (;;) {
				++areaIndex;
				if (areaIndex < manager->getAreasCount()) {
					PArea	area = manager->getArea( areaIndex );
					if (area->getGroup() == group) {
						// found!
						break;;
					}
				} else {
					areaIndex = -1;
					break;
				}
			}
		} else {
			++areaIndex;
			if (areaIndex >= manager->getAreasCount()) {
				areaIndex = -1;
			}
		}
	}

	return (*this);
}


PArea CAreasManager::CAreaIterator::operator *() {
	return manager->getArea( areaIndex );
}


CAreasManager::CAreaIterator CAreasManager::areaBegin(PAreaGroup group) {
	// find first area index
	int areasCount = areas.size();
	int	index;
	for (index = 0; index < areasCount; ++index) {
		if (areas[index]->getGroup() == group) {
			// first area from group found!
			break;
		}
	}
	if (index >= areasCount) {
		// area not found
		return areaEnd( group );
	}

	return CAreaIterator( this, group, index );
}


CAreasManager::CAreaIterator CAreasManager::areaBegin() {
	if (areas.size() == 0) {
		return areaEnd();
	}
	return CAreaIterator( this, NULL, 0 );
}


CAreasManager::CAreaIterator CAreasManager::areaEnd(PAreaGroup group) {
	return CAreaIterator( this, group, -1 );
}


CAreasManager::CAreaIterator CAreasManager::areaEnd() {
	return CAreaIterator( this, NULL, -1 );
}


PArea CAreasManager::autoCreate(const tstring& areaName, PLink creator)
	throw(CLogicException)
{
	if (!creator->canCreateAreas()) {
		throw CLogicException( TEXT("Link ") + creator->getAddress().toString() +
			                   TEXT(" not allowed to create areas.") );
	}
	PAreaGroup	group = CAreasManager::getInstance()->getGroup( creator->getDefaultGroup() );
	if (group == NULL) {
		group = new CAreaGroup();
		group->setName( creator->getDefaultGroup() );
		group->setDescription( TEXT("") );
		CAreasManager::getInstance()->addGroup( group );
	}
	PArea	newArea = new CArea();
	newArea->setGroup(group);

	CAreasManager::getInstance()->addArea( newArea );

	newArea->setName( areaName );

	if (g_vars.m_szAutoDescFileName[0] != _T('\0')) {
		CCfgParser	descParser( g_vars.m_szAutoDescFileName );
		if (!descParser.Failed()) {
			while (!descParser.Eof()) {
				descParser.ReadLine();
				if (descParser.Failed()) {
					if (!descParser.Eof())
						break;
					else
						continue;
				}
				if (descParser.Count() < 3)
					continue;
				if (lstrcmpi( descParser[0], _T("Area") ) == 0 && 
					lstrcmpi( descParser[1], areaName.c_str() ) == 0)
				{
					tstring	desc( descParser[2] );

					for (int i = 0; i < desc.size(); i++) {
						if (desc[i] == _T('_')) {
							desc[i] = _T(' ');
						}
					}
					newArea->setDescription( desc );
					break;
				}
			}
		}
	}

	newArea->setSendPriority( creator->getAutocreatedSendPriority() );
	newArea->setReceivePriority( creator->getAutocreatedReceivePriority() );

	// tie creator and area

	TiePtr	tie = new CTie();
	tie->setLink( creator );
	tie->setRouteType( ROUTE_NONE );

	newArea->addTie( tie );

	// it is possible that creator cannot send, let's change tie's routing in this case

	if (!newArea->canSend( creator )) {
		tie->setRouteType( ROUTE_SEND );
	}

	// form fecho path
	TCHAR	szPath[MAX_PATH];
	TCHAR	szShortName[MAX_PATH];
	TCHAR	goodPath[MAX_PATH];
	if (creator->getAutoCreatedPathName().length() > 0)
		lstrcpy( szPath, creator->getAutoCreatedPathName().c_str() );
	else
		lstrcpy( szPath, g_vars.m_szAutoCreatedPath );

	if (g_vars.m_fLongPaths)
		_stprintf( goodPath, TEXT("%s%s\\"), szPath, areaName.c_str() );
	else {
		int	i = areaName.size();
		if (i > 8)
			i = 8;
		lstrcpyn( szShortName, areaName.c_str(), i+1 );
		for (i = 0; szShortName[i] != TEXT('\0'); i++) {
			if (szShortName[i] == TEXT('.'))
				szShortName[i] = TEXT('_');
		}
		_stprintf( goodPath, TEXT("%s%s\\"), szPath, szShortName );
	}
	newArea->setPath( goodPath );
	if (_taccess( newArea->getWorkingPath().c_str(), 0 ) == 0) {
		int	i = 0;
		do {
			if (g_vars.m_fLongPaths) {
				_stprintf( goodPath, TEXT("%s%s.%03ld\\"), szPath, areaName.c_str(), i );
			}
			else {
				_stprintf( goodPath, TEXT("%s%s.%03lX\\"), szPath, szShortName, i );
			}
			i++;
			newArea->setPath( goodPath );
		} while (_taccess( newArea->getWorkingPath().c_str(), 0 ) == 0);
	}
	MakeDirStructure( newArea->getWorkingPath().c_str() );

	// TODO: add feature to tie some other links, e.g. to tie /101 and /102 when /100
	//       creates area

	CAreasManager::getInstance()->affectAreas();

	return newArea;
}
